/**
 * http.js : utilitaires pour les requtes HTTP mises depuis des scripts.
 *
 * Tir du livre JavaScript  La rfrence, 5e dition,
 * de David Flanagan. Copyright 2006 ditions O'Reilly (ISBN : 2-84177-415-5)
 */

// Vrifier que le module n'a pas dj t charg.
var HTTP;
if (HTTP && (typeof HTTP != "object" || HTTP.NAME))
    throw new Error("L'espace de noms 'HTTP' existe dj");

// Crer notre espace de noms et quelques mta-informations.
HTTP = {};
HTTP.NAME = "HTTP";    // Nom de cet espace de noms.
HTTP.VERSION = 1.0;    // Version de cet espace de noms.

// Voici la liste des fonctions de cration de XMLHttpRequest  essayer.
HTTP._fabriques = [
    function() { return new XMLHttpRequest(); },
    function() { return new ActiveXObject("Msxml2.XMLHTTP"); },
    function() { return new ActiveXObject("Microsoft.XMLHTTP"); }
];

// Lorsquune mthode de fabrique fonctionne, la stocker dans cette variable.
HTTP._fabrique = null;

// Crer et retourner un nouvel objet XMLHttpRequest.
// 
// Au premier appel, essayer la liste des fonctions de fabrique jusqu
// ce que lune delles retourne une valeur non nulle et ne lve pas une
// exception. Lorsque nous avons trouv une mthode qui fonctionne, la
// mmoriser pour son utilisation future.
//
HTTP.nouvelleRequete = function() {
    if (HTTP._fabrique != null) return HTTP._fabrique();

    for(var i = 0; i < HTTP._fabriques.length; i++) {
        try {
            var fabrique = HTTP._fabriques[i];
            var requete = fabrique();
            if (requete != null) {
                HTTP._fabrique = fabrique;
                return requete;
            }
        }
        catch(e) {
            continue;
        }
    }

    // Si nous arrivons ici, cest quaucune des fonctions de fabrique
    // na russit  crer un objet XMLHttpRequest. Nous levons donc une
    // exception pour cet appel et les suivants.
    HTTP._fabrique = function() {
        throw new Error("XMLHttpRequest non pris en charge");
    }
    HTTP._fabrique(); // Gnrer une erreur.
}

/**
 * Utiliser XMLHttpRequest pour obtenir le contenu de lURL indique en
 * utilisant une requte HTTP GET. Lorsque la rponse arrive, elle est 
 * passe, sous forme de texte brut,  la fonction de rappel.
 * 
 * Cette fonction ne bloque pas et na pas de valeur de retour.
 */
HTTP.obtenirTexte = function(url, rappel) {
    var requete = HTTP.nouvelleRequete();
    requete.onreadystatechange = function() {
        if (requete.readyState == 4 && requete.status == 200)
            rappel(requete.responseText);
    }
    requete.open("GET", url);
    requete.send(null);
};

/**
 * Utiliser XMLHttpRequest pour obtenir le contenu de lURL indique en
 * utilisant une requte HTTP GET. Lorsque la rponse arrive, elle est 
 * passe, sous forme d'objet Document XML,  la fonction de rappel.
 * 
 * Cette fonction ne bloque pas et na pas de valeur de retour.
 */
HTTP.obtenirXML = function(url, rappel) {
    var requete = HTTP.nouvelleRequete();
    requete.onreadystatechange = function() {
        if (requete.readyState == 4 && requete.status == 200)
            rappel(requete.responseXML);
    }
    requete.open("GET", url);
    requete.send(null);
};

/**
 * Utiliser une requte HTTP HEAD pour obtenir les en-ttes de lURL
 * indique. Lorsque les en-ttes arrivent, ils sont analyss par
 * HTTP.analyserEntetes() et lobjet rsultant est pass  la fonction
 * de rappel. Si le serveur renvoie un code derreur, la fonction 
 * gererErreur est appele  la place. Si cette fonction nest pas
 * appele, null est pass  la fonction de rappel.
 */
HTTP.obtenirEntetes = function(url, rappel, gererErreur) {
    var requete = HTTP.nouvelleRequete();
    requete.onreadystatechange = function() {
        if (requete.readyState == 4) {
            if (requete.status == 200) {
                rappel(HTTP.analyserEntetes(requete));
            }
            else {
                if (gererErreur) gererErreur(requete.status,
                                             requete.statusText);
                else rappel(null);
            }
        }
    }
    requete.open("HEAD", url);
    requete.send(null);
};

// Analyser les en-ttes de la rponse fournie par un objet XMLHttpRequest
// et retourner les noms et les valeurs des en-ttes comme noms et valeurs
// de proprits dun nouvel objet.
HTTP.analyserEntetes = function(requete) {
    var texteEntete = requete.getAllResponseHeaders();   // Texte venant
                                                         // du serveur.
    var entetes = {}; // Notre valeur de retour.
    var ed = /^\s*/;  // Expression rgulire pour les espaces de dbut.
    var ef = /\s*$/;  // Expression rgulire pour les espaces de fin.

    // Transformer les en-ttes en lignes.
    var lignes = texteEntete.split("\n");
    // Parcourir les lignes.
    for(var i = 0; i < lignes.length; i++) {
        var ligne = lignes[i];
        if (ligne.length == 0) continue;  // Sauter les lignes vides.
        // Couper chaque ligne sur le premier caractre deux-points et
        // supprimer les espaces de dbut et de fin.
        var pos = ligne.indexOf(':');     
        var nom = ligne.substring(0, pos).replace(ed, "").replace(ef, "");
        var valeur = ligne.substring(pos+1).replace(ed, "").replace(ef, "");
        // Stocker la paire nom/valeur de len-tte dans un objet JavaScript.
        entetes[nom] = valeur;
    }
    return entetes;
};

/**
 * Envoyer une requte HTTP POST  lURL indique. Les noms et les valeurs
 * des proprits de lobjet valeurs constituent le corps de la requte.
 * La rponse du serveur est analyse en fonction de son type et la valeur
 * rsultante est passe  la fonction de rappel. Si une erreur HTTP se
 * produit, la fonction indique par gererErreur est invoque ou, si cette
 * fonction nest pas dfinie, null est pass  la fonction de rappel.
 **/
HTTP.post = function(url, valeurs, rappel, gererErreur) {
    var requete = HTTP.nouvelleRequete();
    requete.onreadystatechange = function() {
        if (requete.readyState == 4) {
            if (requete.status == 200) {
                rappel(HTTP._obtenirReponse(requete));
            }
            else {
                if (gererErreur) gererErreur(requete.status,
                                             requete.statusText);
                else rappel(null);
            }
        }
    }

    requete.open("POST", url);
    // Cette en-tte indique au serveur comment interprter le corps
    // de la requte.
    requete.setRequestHeader("Content-Type",
                             "application/x-www-form-urlencoded");
    // Encoder les proprits de lobjet valeurs et les envoyer
    // dans le corps de la requte.
    requete.send(HTTP.coderDonneesFormulaire(valeurs));
};

/**
 * Encoder les paires nom/valeur des proprits dun objet comme
 * si elles provenaient dun formulaire HTML, en utilisant un 
 * format application/x-www-form-urlencoded.
 */
HTTP.encoderDonneesFormulaire = function(donnees) {
    var paires = [];
    var regexp = /%20/g; // Expression rgulire qui correspond
                         //  un espace encod.

    for(var nom in donnees) {
        var valeur = donnees[nom].toString();
        // Crer une paire nom/valeur, en commenant par encoder le nom et
        // la valeur. La fonction globale encodeURIComponent effectue cette 
        // opration, mais elle convertit les espaces en %20 et non en "+",
        // contrairement  ce que nous souhaitons. Nous corrigeons cela grce
        //  la fonction String.replace().
        var paire = encodeURIComponent(nom).replace(regexp,"+") + '=' +
                    encodeURIComponent(valeur).replace(regexp,"+");
        paires.push(paire);
    }

    // Concatner toutes les paires nom/valeur, en les sparant par "&".
    return paires.join('&');
};

/**
 * Analyser une rponse HTTP en fonction de son en-tte Content-Type et
 * retourner l'objet obtenu.
 */
HTTP._obtenirReponse = function(requete) {
    // Tester le type du contenu retourn par le serveur.
    switch(requete.getResponseHeader("Content-Type")) {
    case "text/xml":
        // Sil sagit dun document XML, utilisez lobjet Document cr.
        return requete.responseXML;

    case "text/json":
    case "text/javascript":
    case "application/javascript":
    case "application/x-javascript":
        // Si la rponse contient du code JavaScript ou une valeur au 
        // format JSON, appeler eval() sur le texte afin de le convertir
        // en valeur JavaScript. Note : neffectuer cette opration que
        // si le code JavaScript provient dun serveur de confiance !
        return eval(requete.responseText);

    default:
        // Sinon, traiter la rponse comme du texte brut et la retourner
        // sous forme dune chane de caractres.
        return requete.responseText;
    }
};

/**
 * Envoyer une requte HTTP GET pour lURL indique. En cas de succs, la
 * rponse est convertie en un objet dpendant de len-tte Content-Type,
 * pass ensuite  la fonction de rappel indique. Dautres arguments
 * peuvent tre passs comme des proprits de lobjet options.
 *
 * Si la rponse signale une erreur (par exemple, une erreur 404 Not Found),
 * le code et le message dtat sont passs  la fonction options.gererErreur.
 * Si le gestionnaire derreur nest pas prcis, la fonction de rappel est
 * appele avec largument null.
 * 
 * Si lobjet options.parametres est dfini, ses proprits sont traites
 * comme les noms et les valeurs des paramtres de la requte. Elles sont
 * converties en une chane dURL avec HTTP.coderDonneesFormulaire() et
 * ajoutes  lURL aprs un '?'.
 * 
 * Si une fonction options.gestionnaireProgression est indique, elle est
 * appele chaque fois que la proprit readyState prend une valeur
 * infrieure  4. Chaque appel de la fonction de gestion de la progression
 * reoit un entier indiquant le nombre de fois quelle a t appele.
 *
 * Si une valeur options.temporisation est spcifie, la requte est
 * annule si elle ne sest pas termine avant lchance de la dure 
 * indique en millisecondes. Si le dlai dattente est coul et si
 * options.gestionnaireTemporisation est dfinie, cette fonction est 
 * appele avec lURL demande en argument.
 **/
HTTP.get = function(url, rappel, options) {
    var requete = HTTP.nouvelleRequete();
    var n = 0;
    var minuterie;
    if (options.temporisation)
        minuterie = setTimeout(
                           function() {
                               requete.abort();
                               if (options.gestionnaireTemporisation)
                                   options.gestionnaireTemporisation(url);
                           },
                           options.temporisation);

    requete.onreadystatechange = function() {
        if (requete.readyState == 4) {
            if (minuterie) clearTimeout(minuterie);
            if (requete.status == 200) {
                rappel(HTTP._obtenirReponse(requete));
            }
            else {
                if (options.gererErreur)
                    options.gererErreur(requete.status,
                                        requete.statusText);
                else rappel(null);
            }
        }
        else if (options.gestionnaireProgression) {
            options.gestionnaireProgression(++n);
        }
    }

    var cible = url;
    if (options.parametres)
        cible += "?" + HTTP.coderDonneesFormulaire(options.parametres)
    requete.open("GET", cible);
    requete.send(null);
};

HTTP.obtenirTexteAvecScript = function(url, rappel) {
    // Crer un nouvel lment <script> et lajouter au document.
    var script = document.createElement("script");
    document.body.appendChild(script);

    // Obtenir un nom de fonction unique.
    var nomFonction = "fonction" + HTTP.obtenirTexteAvecScript.compteur++;

    // Dfinir une fonction de ce nom, dont lespace de noms est la fonction
    // courante. Le script gnr sur le serveur invoque cette fonction.
    HTTP.obtenirTexteAvecScript[nomFonction] = function(texte) {
        // Passer le texte  la fonction de rappel.
        rappel(texte);

        // Effacer la balise <script> et la fonction gnre.
        document.body.removeChild(script);
        delete HTTP.obtenirTexteAvecScript[nomFonction];
    }

    // Coder lURL  charger et le nom de la fonction et placer le rsultat
    // dans les arguments du script cot serveur jsquoter.php. Fixer la
    // proprit src de la balise <script> afin de charger lURL.
    script.src = "jsquoter.php" +
                 "?url=" + encodeURIComponent(url) + "&fonction=" +
                 encodeURIComponent("HTTP.obtenirTexteAvecScript." +
                                    nomFonction);
}

// Nous utilisons cette variable pour gnrer des fonctions de rappel
// aux noms uniques, pour le cas o plusieurs requtes seraient mises 
// en mme temps.
HTTP.obtenirTexteAvecScript.compteur = 0;
